// Global variables
let currentStep = 0;
let currentQuestionIndex = 0;
let selectedAnswer = null;
let score = 0;
let quizAnswers = [];
let isAnimating = false;
let currentAnimationStep = 0;
let animationInterval = null;
let selectedMaterials = [];
let activityStep = 0;
let isSimulationRunning = false;
let simulationInterval = null;
let currentAngle = 45;
let currentMaterial = 'glass';
let selectedConcept = 0;
let selectedFact = 0;
let selectedUseCase = 0;
let selectedAngleConcept = 0;
let checkedPoints = [];

// Quiz questions
const quizQuestions = [
    {
        question: "What is refraction of light?",
        options: [
            "Light bouncing off a surface",
            "Light bending when it passes from one transparent medium to another",
            "Light getting brighter in intensity",
            "Light completely disappearing"
        ],
        correct: 1,
        explanation: "Refraction is the phenomenon where light rays change direction (bend) when passing from one transparent medium to another with different optical densities."
    },
    {
        question: "What is the typical shape of an optical prism used for light dispersion?",
        options: [
            "Circular",
            "Square",
            "Triangular",
            "Hexagonal"
        ],
        correct: 2,
        explanation: "An optical prism typically has a triangular cross-section with flat, polished surfaces that are specifically designed to refract and disperse light effectively."
    },
    {
        question: "How many colors does white light split into when passing through a prism, and what is the correct order?",
        options: [
            "5 colors in ROYGB order",
            "6 colors in VIBGYO order",
            "7 colors in VIBGYOR order",
            "8 colors in ROYGBIV order"
        ],
        correct: 2,
        explanation: "White light splits into 7 colors in VIBGYOR order: Violet, Indigo, Blue, Green, Yellow, Orange, and Red. This sequence represents the visible light spectrum."
    },
    {
        question: "Which color in the VIBGYOR spectrum bends the LEAST when passing through a prism?",
        options: [
            "Red",
            "Blue",
            "Green",
            "Violet"
        ],
        correct: 0,
        explanation: "Red light has the longest wavelength and the lowest refractive index in glass, so it bends the least. Violet light bends the most due to its shorter wavelength and higher refractive index."
    },
    {
        question: "What is the scientific term for the splitting of white light into its component colors?",
        options: [
            "Reflection",
            "Absorption",
            "Dispersion",
            "Transmission"
        ],
        correct: 2,
        explanation: "Dispersion is the process where white light separates into its constituent colors (VIBGYOR) when passing through a prism due to different refractive indices for different wavelengths."
    },
    {
        question: "What creates natural rainbows in the sky after rain?",
        options: [
            "Sunlight refracting and reflecting through water droplets in the air",
            "Clouds changing their natural color",
            "The sun moving closer to Earth temporarily",
            "Atmospheric pressure changes"
        ],
        correct: 0,
        explanation: "Rainbows are created when sunlight undergoes refraction, dispersion, and internal reflection through millions of tiny water droplets suspended in the air, each acting like a miniature prism!"
    },
    {
        question: "According to Snell's Law, what happens to light when it enters a material with a higher refractive index?",
        options: [
            "It bends away from the normal",
            "It bends towards the normal",
            "It travels in a straight line",
            "It completely stops"
        ],
        correct: 1,
        explanation: "When light enters a material with a higher refractive index (denser medium), it slows down and bends towards the normal line according to Snell's Law: n₁sin(θ₁) = n₂sin(θ₂)."
    }
];

// Activity steps
const activitySteps = [
    {
        title: "Step 1: Gather your materials",
        content: "Select the materials you have available:",
        type: "materials"
    },
    {
        title: "Step 2: Set up the experiment",
        content: "1. Fill a clear glass with clean water<br/>2. Place the small mirror inside the glass at a 45-degree angle<br/>3. Position the setup near a white wall or place a white sheet of paper nearby to act as a screen",
        type: "setup"
    },
    {
        title: "Step 3: Shine the light",
        content: "Shine your flashlight at the mirror through the water. Adjust the angle to find the best position for creating a rainbow!",
        type: "light"
    },
    {
        title: "Step 4: Observe the rainbow!",
        content: "Look for a small rainbow on the wall! You've just created dispersion of light - the same physics that makes rainbows in the sky after rain!",
        type: "observe"
    }
];

// Key learning points for summary
const keyPoints = [
    {
        title: "Refraction is the bending of light",
        description: "When light travels from one transparent medium to another with different optical densities, it changes direction due to the change in speed",
        icon: "🔄"
    },
    {
        title: "A prism is a transparent triangular optical element",
        description: "Made of glass or plastic with flat, polished surfaces arranged at precise angles that can refract and disperse light effectively",
        icon: "🔺"
    },
    {
        title: "White light contains all colors in VIBGYOR order",
        description: "What appears as colorless white light is actually composed of seven distinct colors: Violet, Indigo, Blue, Green, Yellow, Orange, and Red",
        icon: "🎨"
    },
    {
        title: "Dispersion creates rainbow spectra",
        description: "Different colors have different wavelengths and refractive indices, causing them to bend by different amounts and separate into a visible spectrum",
        icon: "🌈"
    },
    {
        title: "Violet bends most, red bends least",
        description: "Colors have different wavelengths and refractive indices - violet (shortest wavelength) bends the most, while red (longest wavelength) bends the least",
        icon: "📏"
    },
    {
        title: "Refraction occurs everywhere around us",
        description: "From natural rainbows and eyeglasses to cameras, diamonds, telescopes, and even fiber optic internet cables!",
        icon: "👁️"
    },
    {
        title: "Angles and Snell's Law govern refraction",
        description: "The relationship between incident and refraction angles follows Snell's Law: n₁sin(θ₁) = n₂sin(θ₂), where n represents refractive indices",
        icon: "📐"
    },
    {
        title: "You can create rainbows at home",
        description: "Using simple materials like water, mirrors, and light sources, you can demonstrate dispersion and create your own rainbow through refraction",
        icon: "🏠"
    }
];

// Real life use cases
const useCases = [
    {
        title: "Rainbows in the Sky",
        description: "Rainbows are made by sunlight refracting through tiny water droplets in the air after rain!",
        icon: "🌈",
        detail: "Each water droplet acts like a tiny prism, splitting sunlight into all the beautiful colors we see."
    },
    {
        title: "Eyeglasses & Contact Lenses",
        description: "Glasses use refraction to help people see better by bending light to focus properly on the retina.",
        icon: "👓",
        detail: "Different lens shapes can correct nearsightedness, farsightedness, and astigmatism."
    },
    {
        title: "Cameras & Photography",
        description: "Camera lenses use multiple prisms and curved glass to focus light and create sharp, clear pictures.",
        icon: "📷",
        detail: "Professional cameras have complex lens systems with many glass elements to control light precisely."
    },
    {
        title: "Binoculars & Telescopes",
        description: "These tools use prisms to flip images right-side up and magnify distant objects for better viewing.",
        icon: "🔭",
        detail: "Without prisms, the images in binoculars would appear upside down!"
    },
    {
        title: "Diamonds & Jewelry",
        description: "Diamonds are cut in special ways to create maximum light refraction, making them sparkle brilliantly.",
        icon: "💎",
        detail: "The more faces (facets) a diamond has, the more it can split and reflect light, creating more sparkle."
    },
    {
        title: "Fiber Optic Cables",
        description: "Internet cables use refraction to send light signals carrying data over long distances at the speed of light.",
        icon: "🌐",
        detail: "Your internet connection might be traveling through glass fibers using total internal refraction!"
    }
];

// Materials for simulation
const materials = [
    { name: 'glass', index: 1.5, color: 'rgba(59, 130, 246, 0.3)' },
    { name: 'diamond', index: 2.42, color: 'rgba(236, 72, 153, 0.3)' },
    { name: 'water', index: 1.33, color: 'rgba(6, 182, 212, 0.3)' },
    { name: 'plastic', index: 1.4, color: 'rgba(34, 197, 94, 0.3)' }
];

// Initialize the website
document.addEventListener('DOMContentLoaded', function() {
    initializeNavigation();
    initializeQuiz();
    initializeSummary();
    updateActiveNavigation();
    
    // Set up scroll listener for navigation
    window.addEventListener('scroll', updateActiveNavigation);
});

// Navigation functions
function initializeNavigation() {
    const navLinks = document.querySelectorAll('.nav-link');
    navLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            const targetId = this.getAttribute('href').substring(1);
            scrollToSection(targetId);
        });
    });
}

function scrollToSection(sectionId) {
    const element = document.getElementById(sectionId);
    if (element) {
        element.scrollIntoView({ behavior: 'smooth' });
    }
}

function updateActiveNavigation() {
    const sections = ['home', 'refraction', 'prism', 'light-travel', 'why-happens', 'real-life', 'activity', 'simulation', 'angles-laws', 'quiz', 'summary', 'contact'];
    const scrollPosition = window.scrollY + 100;
    
    for (const section of sections) {
        const element = document.getElementById(section);
        if (element) {
            const { offsetTop, offsetHeight } = element;
            if (scrollPosition >= offsetTop && scrollPosition < offsetTop + offsetHeight) {
                // Update active navigation
                document.querySelectorAll('.nav-link').forEach(link => {
                    link.classList.remove('active');
                    if (link.getAttribute('href') === `#${section}`) {
                        link.classList.add('active');
                    }
                });
                break;
            }
        }
    }
}

// Refraction demo functions
function toggleRefractionDemo() {
    const pencil = document.getElementById('pencil');
    const lightRays = document.getElementById('lightRays');
    const normalLine = document.getElementById('normalLine');
    const instruction = document.getElementById('demoInstruction');
    
    if (pencil.style.transform === 'translateX(-50%) rotate(12deg)') {
        // Reset
        pencil.style.transform = 'translateX(-50%)';
        lightRays.classList.remove('visible');
        normalLine.classList.remove('visible');
        instruction.textContent = 'Click to see refraction in action';
    } else {
        // Show refraction
        pencil.style.transform = 'translateX(-50%) rotate(12deg)';
        lightRays.classList.add('visible');
        normalLine.classList.add('visible');
        instruction.textContent = 'Light rays bending as they pass through water!';
    }
}

// Prism section functions
function selectFact(index) {
    document.querySelectorAll('.fact-item').forEach((item, i) => {
        item.classList.toggle('active', i === index);
    });
    selectedFact = index;
}

// Light travel animation functions
function toggleAnimation() {
    const playBtn = document.querySelector('.play-btn');
    const playText = document.getElementById('playText');
    
    if (isAnimating) {
        stopAnimation();
        playText.textContent = '▶️ Play';
    } else {
        startAnimation();
        playText.textContent = '⏸️ Pause';
    }
    isAnimating = !isAnimating;
}

function startAnimation() {
    currentAnimationStep = 0;
    animationInterval = setInterval(() => {
        updateAnimationStep();
        currentAnimationStep = (currentAnimationStep + 1) % 5;
    }, 2500);
    updateAnimationStep();
}

function stopAnimation() {
    if (animationInterval) {
        clearInterval(animationInterval);
        animationInterval = null;
    }
}

function updateAnimationStep() {
    const steps = document.querySelectorAll('.step');
    const incomingLight = document.getElementById('incomingLight');
    const dispersedColors = document.getElementById('dispersedColors');
    const stepIndicator = document.getElementById('stepIndicator');
    
    // Update step highlighting
    steps.forEach((step, index) => {
        step.classList.toggle('active', index === currentAnimationStep);
    });
    
    // Update animation elements
    switch (currentAnimationStep) {
        case 0:
            incomingLight.classList.remove('animate');
            dispersedColors.classList.remove('visible');
            stepIndicator.textContent = 'White light approaches the prism surface';
            break;
        case 1:
            incomingLight.classList.add('animate');
            stepIndicator.textContent = 'Light slows down and bends at the first surface (refraction occurs)';
            break;
        case 2:
            stepIndicator.textContent = 'Light travels through the glass prism at reduced speed';
            break;
        case 3:
            stepIndicator.textContent = 'Light bends again as it exits the prism into air';
            break;
        case 4:
            dispersedColors.classList.add('visible');
            stepIndicator.textContent = 'Light disperses into seven beautiful colors in VIBGYOR order!';
            break;
    }
}

// Why it happens section functions
function selectConcept(index) {
    document.querySelectorAll('.concept-tab').forEach((tab, i) => {
        tab.classList.toggle('active', i === index);
    });
    selectedConcept = index;
}

// Real life uses functions
function selectUseCase(index) {
    document.querySelectorAll('.use-case').forEach((item, i) => {
        item.classList.toggle('active', i === index);
    });
    
    const detailContent = document.getElementById('useCaseDetail');
    const useCase = useCases[index];
    
    detailContent.innerHTML = `
        <span class="detail-icon">${useCase.icon}</span>
        <h3>${useCase.title}</h3>
        <p>${useCase.description}</p>
        <div class="detail-info">
            <strong>Did you know?</strong> ${useCase.detail}
        </div>
    `;
    
    selectedUseCase = index;
}

// Activity section functions
function toggleMaterial(element) {
    element.classList.toggle('selected');
    updateActivityProgress();
}

function updateActivityProgress() {
    const selectedMaterials = document.querySelectorAll('.material.selected');
    const progress = document.getElementById('activityProgress');
    const nextBtn = document.getElementById('nextBtn');
    
    if (selectedMaterials.length >= 3) {
        progress.style.width = '50%';
        nextBtn.disabled = false;
    } else {
        progress.style.width = '25%';
        nextBtn.disabled = true;
    }
}

function nextStep() {
    if (activityStep < activitySteps.length - 1) {
        activityStep++;
        updateActivityStep();
    }
}

function resetActivity() {
    activityStep = 0;
    document.querySelectorAll('.material').forEach(material => {
        material.classList.remove('selected');
    });
    updateActivityStep();
    updateActivityProgress();
}

function updateActivityStep() {
    const stepContent = document.getElementById('stepContent');
    const progress = document.getElementById('activityProgress');
    const nextBtn = document.getElementById('nextBtn');
    const step = activitySteps[activityStep];
    
    // Update progress
    progress.style.width = `${((activityStep + 1) / activitySteps.length) * 100}%`;
    
    // Update content
    let content = `<h4>${step.title}</h4><p>${step.content}</p>`;
    
    if (step.type === 'materials') {
        content += `
            <div class="materials">
                <div class="material" onclick="toggleMaterial(this)">
                    <span class="material-icon">🥛</span>
                    <span>Glass of Water</span>
                </div>
                <div class="material" onclick="toggleMaterial(this)">
                    <span class="material-icon">🪞</span>
                    <span>Small Mirror</span>
                </div>
                <div class="material" onclick="toggleMaterial(this)">
                    <span class="material-icon">🔦</span>
                    <span>Flashlight</span>
                </div>
            </div>
        `;
    }
    
    stepContent.innerHTML = content;
    
    // Update experiment visualization
    updateExperimentVisualization();
    
    // Update next button
    if (activityStep === activitySteps.length - 1) {
        nextBtn.style.display = 'none';
    } else {
        nextBtn.style.display = 'block';
        nextBtn.disabled = activityStep === 0;
    }
}

function updateExperimentVisualization() {
    const flashlight = document.getElementById('flashlight');
    const lightBeam = document.getElementById('lightBeam');
    const rainbow = document.getElementById('rainbowExp');
    const overlay = document.getElementById('instructionOverlay');
    
    switch (activityStep) {
        case 0:
            flashlight.classList.remove('visible');
            lightBeam.classList.remove('visible');
            rainbow.classList.remove('visible');
            overlay.textContent = 'Gather your materials to begin the experiment';
            break;
        case 1:
            flashlight.classList.add('visible');
            overlay.textContent = 'Set up your glass with water and position the mirror';
            break;
        case 2:
            lightBeam.classList.add('visible');
            overlay.textContent = 'Shine the flashlight at the mirror through the water';
            break;
        case 3:
            rainbow.classList.add('visible');
            overlay.textContent = '🌈 Success! You\'ve created a rainbow using refraction!';
            break;
    }
}

// Simulation section functions
function toggleSimulation() {
    const playText = document.getElementById('simPlayText');
    const status = document.getElementById('simStatus');
    
    if (isSimulationRunning) {
        stopSimulation();
        playText.textContent = '▶️ Start';
        status.textContent = 'Paused';
    } else {
        startSimulation();
        playText.textContent = '⏸️ Pause';
        status.textContent = 'Running';
    }
}

function startSimulation() {
    isSimulationRunning = true;
    const incidentRay = document.getElementById('incidentRay');
    const refractedRays = document.getElementById('refractedRays');
    
    incidentRay.classList.add('visible');
    setTimeout(() => {
        refractedRays.classList.add('visible');
    }, 1000);
    
    updateSimulationAngles();
}

function stopSimulation() {
    isSimulationRunning = false;
    const incidentRay = document.getElementById('incidentRay');
    const refractedRays = document.getElementById('refractedRays');
    
    incidentRay.classList.remove('visible');
    refractedRays.classList.remove('visible');
}

function resetSimulation() {
    stopSimulation();
    currentAngle = 45;
    currentMaterial = 'glass';
    
    document.getElementById('angleSlider').value = 45;
    document.getElementById('angleValue').textContent = '45';
    document.getElementById('speedSlider').value = 1;
    document.getElementById('speedValue').textContent = '1';
    
    document.querySelectorAll('.material-btn').forEach(btn => {
        btn.classList.remove('active');
        if (btn.textContent.includes('Glass')) {
            btn.classList.add('active');
        }
    });
    
    updateSimulationDisplay();
}

function updateAngle(value) {
    currentAngle = parseInt(value);
    document.getElementById('angleValue').textContent = value;
    document.getElementById('currentAngle').textContent = value + '°';
    updateSimulationAngles();
}

function updateSpeed(value) {
    document.getElementById('speedValue').textContent = value;
}

function selectMaterial(material) {
    currentMaterial = material;
    document.querySelectorAll('.material-btn').forEach(btn => {
        btn.classList.remove('active');
    });
    event.target.classList.add('active');
    
    const materialData = materials.find(m => m.name === material);
    document.getElementById('currentMaterial').textContent = `${material.charAt(0).toUpperCase() + material.slice(1)} (n = ${materialData.index})`;
    document.getElementById('materialLabel').textContent = `${material.charAt(0).toUpperCase() + material.slice(1)} (n = ${materialData.index})`;
    
    updateSimulationDisplay();
    updateSimulationAngles();
}

function toggleRays() {
    const showRays = document.getElementById('showRays').checked;
    const incidentRay = document.getElementById('incidentRay');
    const refractedRays = document.getElementById('refractedRays');
    
    if (showRays && isSimulationRunning) {
        incidentRay.classList.add('visible');
        refractedRays.classList.add('visible');
    } else {
        incidentRay.classList.remove('visible');
        refractedRays.classList.remove('visible');
    }
}

function updateSimulationDisplay() {
    const prism = document.getElementById('simPrism');
    const materialData = materials.find(m => m.name === currentMaterial);
    
    if (prism && materialData) {
        prism.style.background = `linear-gradient(135deg, ${materialData.color}, rgba(147, 51, 234, 0.3))`;
    }
}

function updateSimulationAngles() {
    const incidentRay = document.getElementById('incidentRay');
    const materialData = materials.find(m => m.name === currentMaterial);
    
    if (incidentRay) {
        incidentRay.style.transform = `translateY(-50%) rotate(${-currentAngle}deg)`;
    }
    
    // Calculate refraction angle using Snell's law
    const n1 = 1.0; // air
    const n2 = materialData ? materialData.index : 1.5;
    const incidentRad = (currentAngle * Math.PI) / 180;
    const refractionRad = Math.asin((n1 * Math.sin(incidentRad)) / n2);
    const refractionAngle = (refractionRad * 180) / Math.PI;
    
    document.getElementById('incidentAngleLabel').textContent = currentAngle + '°';
    document.getElementById('refractionAngleLabel').textContent = refractionAngle.toFixed(1) + '°';
    
    // Update color analysis
    const colors = ['violet', 'indigo', 'blue', 'green', 'yellow', 'orange', 'red'];
    const baseAngles = [29.2, 28.8, 28.4, 28.0, 27.8, 27.6, 27.4];
    
    colors.forEach((color, index) => {
        const element = document.getElementById(color + 'Angle');
        if (element) {
            const adjustedAngle = baseAngles[index] * (refractionAngle / 28.1);
            element.textContent = adjustedAngle.toFixed(1) + '°';
        }
    });
}

// Angles and Laws section functions
function selectAngleConcept(index) {
    document.querySelectorAll('.concept-item').forEach((item, i) => {
        item.classList.toggle('active', i === index);
    });
    
    const snellsLawCard = document.getElementById('snellsLawCard');
    if (snellsLawCard) {
        snellsLawCard.style.display = index === 2 ? 'flex' : 'none';
    }
    
    selectedAngleConcept = index;
}

function updateCalculatorAngle(value) {
    currentAngle = parseInt(value);
    document.getElementById('calcAngleValue').textContent = value;
    document.getElementById('calcIncidentLabel').textContent = value + '°';
    
    const calcIncidentRay = document.getElementById('calcIncidentRay');
    if (calcIncidentRay) {
        calcIncidentRay.style.transform = `translateY(-50%) rotate(${-value}deg)`;
    }
    
    updateCalculatorRefraction();
}

function selectCalcMaterial(material, index) {
    document.querySelectorAll('.calc-material-btn').forEach(btn => {
        btn.classList.remove('active');
    });
    event.target.classList.add('active');
    
    document.getElementById('calcMaterialLabel').textContent = `${material.charAt(0).toUpperCase() + material.slice(1)} (n = ${index})`;
    
    updateCalculatorRefraction();
}

function updateCalculatorRefraction() {
    const incidentAngle = parseInt(document.getElementById('calcAngleSlider').value);
    const materialIndex = 1.5; // Default to glass
    
    // Calculate refraction angle
    const n1 = 1.0;
    const n2 = materialIndex;
    const incidentRad = (incidentAngle * Math.PI) / 180;
    const refractionRad = Math.asin((n1 * Math.sin(incidentRad)) / n2);
    const refractionAngle = (refractionRad * 180) / Math.PI;
    
    document.getElementById('calcRefractionLabel').textContent = refractionAngle.toFixed(1) + '°';
    
    const calcRefractedRay = document.getElementById('calcRefractedRay');
    if (calcRefractedRay) {
        calcRefractedRay.style.transform = `translateY(-50%) rotate(${refractionAngle}deg)`;
    }
}

// Quiz functions
function initializeQuiz() {
    currentQuestionIndex = 0;
    score = 0;
    quizAnswers = [];
    selectedAnswer = null;
    
    document.getElementById('totalQuestions').textContent = quizQuestions.length;
    loadQuestion();
}

function loadQuestion() {
    const question = quizQuestions[currentQuestionIndex];
    
    document.getElementById('currentQuestionNum').textContent = currentQuestionIndex + 1;
    document.getElementById('questionText').textContent = question.question;
    
    const optionsContainer = document.getElementById('quizOptions');
    optionsContainer.innerHTML = '';
    
    question.options.forEach((option, index) => {
        const button = document.createElement('button');
        button.className = 'quiz-option';
        button.onclick = () => selectAnswer(index);
        button.innerHTML = `
            <span class="option-letter">${String.fromCharCode(65 + index)}</span>
            <span class="option-text">${option}</span>
        `;
        optionsContainer.appendChild(button);
    });
    
    // Update progress
    const progressFill = document.getElementById('quizProgressFill');
    progressFill.style.width = `${((currentQuestionIndex + 1) / quizQuestions.length) * 100}%`;
    
    // Hide explanation and reset button state
    document.getElementById('quizExplanation').style.display = 'none';
    document.getElementById('nextQuestionBtn').disabled = true;
    selectedAnswer = null;
}

function selectAnswer(answerIndex) {
    if (selectedAnswer !== null) return; // Prevent multiple selections
    
    selectedAnswer = answerIndex;
    const question = quizQuestions[currentQuestionIndex];
    const isCorrect = answerIndex === question.correct;
    
    // Update score
    if (isCorrect) {
        score++;
    }
    
    // Update display
    document.getElementById('currentScore').textContent = score;
    document.getElementById('questionsAnswered').textContent = currentQuestionIndex + 1;
    
    // Style the options
    const options = document.querySelectorAll('.quiz-option');
    options.forEach((option, index) => {
        option.style.pointerEvents = 'none';
        if (index === question.correct) {
            option.classList.add('correct');
        } else if (index === answerIndex && index !== question.correct) {
            option.classList.add('incorrect');
        }
    });
    
    // Show explanation
    showExplanation(isCorrect, question.explanation);
    
    // Enable next button
    document.getElementById('nextQuestionBtn').disabled = false;
}

function showExplanation(isCorrect, explanationText) {
    const explanationDiv = document.getElementById('quizExplanation');
    const explanationIcon = document.getElementById('explanationIcon');
    const explanationTitle = document.getElementById('explanationTitle');
    const explanationTextDiv = document.getElementById('explanationText');
    
    explanationIcon.textContent = isCorrect ? '🎉' : '❌';
    explanationTitle.textContent = isCorrect ? 'Correct!' : 'Incorrect';
    explanationTextDiv.textContent = explanationText;
    
    if (!isCorrect) {
        explanationDiv.classList.add('incorrect');
    } else {
        explanationDiv.classList.remove('incorrect');
    }
    
    explanationDiv.style.display = 'block';
}

function nextQuestion() {
    if (currentQuestionIndex < quizQuestions.length - 1) {
        currentQuestionIndex++;
        loadQuestion();
    } else {
        showQuizResults();
    }
}

function showQuizResults() {
    document.getElementById('quizContainer').style.display = 'none';
    document.getElementById('quizComplete').style.display = 'block';
    
    const percentage = Math.round((score / quizQuestions.length) * 100);
    
    document.getElementById('finalScore').textContent = `${score}/${quizQuestions.length}`;
    document.getElementById('scorePercentage').textContent = `You scored ${percentage}%`;
    document.getElementById('correctAnswers').textContent = score;
    document.getElementById('incorrectAnswers').textContent = quizQuestions.length - score;
    
    // Set score message
    let message;
    if (percentage === 100) {
        message = "🎉 Perfect! You're a light refraction expert!";
    } else if (percentage >= 85) {
        message = "🌟 Excellent! You really understand light and prisms!";
    } else if (percentage >= 70) {
        message = "👍 Good job! You're getting the hang of it!";
    } else if (percentage >= 50) {
        message = "📚 Not bad! Try reviewing the material and take the quiz again.";
    } else {
        message = "🤔 Keep studying! Light and prisms are tricky but fascinating topics.";
    }
    
    document.getElementById('scoreMessage').textContent = message;
}

function restartQuiz() {
    document.getElementById('quizComplete').style.display = 'none';
    document.getElementById('quizContainer').style.display = 'block';
    initializeQuiz();
}

// Summary section functions
function initializeSummary() {
    const keyPointsContainer = document.getElementById('keyPoints');
    checkedPoints = new Array(keyPoints.length).fill(false);
    
    keyPointsContainer.innerHTML = '';
    
    keyPoints.forEach((point, index) => {
        const pointElement = document.createElement('div');
        pointElement.className = 'key-point';
        pointElement.onclick = () => toggleKeyPoint(index);
        
        pointElement.innerHTML = `
            <div class="check-button" id="check-${index}">
                <span style="display: none;">✓</span>
            </div>
            <div class="point-content">
                <div class="point-header">
                    <span class="point-icon">${point.icon}</span>
                    <h4 class="point-title">${point.title}</h4>
                </div>
                <p class="point-description">${point.description}</p>
            </div>
        `;
        
        keyPointsContainer.appendChild(pointElement);
    });
    
    updateSummaryProgress();
}

function toggleKeyPoint(index) {
    checkedPoints[index] = !checkedPoints[index];
    
    const pointElement = document.querySelectorAll('.key-point')[index];
    const checkButton = document.getElementById(`check-${index}`);
    const checkMark = checkButton.querySelector('span');
    
    if (checkedPoints[index]) {
        pointElement.classList.add('checked');
        checkMark.style.display = 'block';
    } else {
        pointElement.classList.remove('checked');
        checkMark.style.display = 'none';
    }
    
    updateSummaryProgress();
}

function updateSummaryProgress() {
    const completedCount = checkedPoints.filter(Boolean).length;
    const totalPoints = keyPoints.length;
    const percentage = (completedCount / totalPoints) * 100;
    
    document.getElementById('completedCount').textContent = completedCount;
    document.getElementById('totalPoints').textContent = totalPoints;
    document.getElementById('summaryProgress').style.width = `${percentage}%`;
    
    const celebration = document.getElementById('completionCelebration');
    if (completedCount === totalPoints) {
        celebration.style.display = 'block';
    } else {
        celebration.style.display = 'none';
    }
}

// Initialize everything when the page loads
document.addEventListener('DOMContentLoaded', function() {
    // Initialize all sections
    initializeNavigation();
    initializeQuiz();
    initializeSummary();
    updateActiveNavigation();
    
    // Initialize default states
    selectFact(0);
    selectUseCase(0);
    selectConcept(0);
    selectAngleConcept(0);
    updateActivityStep();
    updateSimulationDisplay();
    updateCalculatorRefraction();
    
    // Set up scroll listener
    window.addEventListener('scroll', updateActiveNavigation);
    
    // Initialize activity step
    activityStep = 0;
    updateActivityStep();
});